package mcfall.raytracer;

import java.awt.Dimension;
import java.util.List;

import mcfall.math.ColumnVector;
import mcfall.math.IncompatibleMatrixException;
import mcfall.math.Matrix;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.math.Vector;
import mcfall.raytracer.objects.GenericPlane;
import mcfall.raytracer.objects.HitRecord;

import org.apache.log4j.or.ObjectRenderer;

/**
 * The Camera class represents a particular way of looking at world.  The rectangular area that can be seen through the
 * camera is called its <i>view plane</i>.  The camera is determined by several factors:
 * <ul>
 *   <li> Its location in space (its "eye") </li>
 *   <li> A point that the camera is "looking at"</li>
 *   <li> The orientation of the camera (how it is rotated while looking at a particular point) </li>
 *   <li> The "view angle" and a desired aspect ratio, which determines how much of the world can be seen  </li>
 *   <li> The distance from the "eye" to the view plane</li>
 * </ul>
 * @author mcfall
 *
 */
public class Camera implements ObjectRenderer {
	private Vector u;
	private Vector v;
	private Vector n;
	
	private Point eye;
	private Point lookAt;
	
	private double aspect;
	private double theta;
	
	private double W;
	private double H;
	private double N;
	
	private Dimension planeSize;
	
	/**
	 * Constructs a camera given its location, a point it is looking at, the viewable angle and aspect ratio of the camera's
	 * plane of projection, and the distance to the view plane
	 * @param location A Point describing the location of the camera
	 * @param lookAt A point describing the point the camera is looking at; this point is the center of the view plane
	 * @param viewAngle an angle, in degrees, specifying the total view angle of the camera
	 * @param aspectRatio the ratio of the plane of projection's width to its height
	 * @param distance the distance the plane of projection lies from the eye
	 * @param planeSize the number of rows and columns that make up the 
	 */
	
	public Camera (Point location, Point lookAt, double viewAngle, double aspectRatio, double distance, Dimension planeSize) {
		this.eye = location;
		this.lookAt = lookAt;
		
		aspect = aspectRatio;
		N = distance;
		this.planeSize = new Dimension(planeSize);
		theta = viewAngle;
		
		H = N * Math.tan(Math.toRadians(viewAngle/2.0));
		W = H*aspect;
		
		double[] nValues = new double[4];
		nValues[0] = location.getX()-lookAt.getX();
		nValues[1] = location.getY()-lookAt.getY();
		nValues[2] = location.getZ()-lookAt.getZ();
		
		double[] upValues = new double[4];
		upValues[1] = 1.0;
		ColumnVector up = new ColumnVector (4, upValues);
				
		n = new ColumnVector(4, nValues);
		u = up.cross(n);	
		v = n.cross(u);
		u = u.normalize();
		n = n.normalize();
		v = v.normalize();
		
	}
	
	/**
	 * Creates a ray which goes through a particular point on the view plane
	 * @param column the column of the view plane, in the range through getPlaneWidth()-1
	 * @param row the row of the view plane, in the range 0 through getPlaneHeight()-1	 
	 * @return a ray which passes through the (column, row) block of the view plane.  The starting
	 * point of the Ray is the camera's eye, and point when <i>t</i>=1 is located on the view plane
	 */
	public Ray rayThrough (int column, int row) {		
		Vector nPart = Vector.fromColumnMatrix(n.scalarMultiply(-N));

		Vector uPart = Vector.fromColumnMatrix(u.scalarMultiply(W*(2.0*column/getPlaneWidth()-1)));
		Vector vPart = Vector.fromColumnMatrix(v.scalarMultiply(H*(2.0*row/getPlaneHeight()-1)));

		Vector direction = nPart.add(uPart).add(vPart);

		return new Ray (eye, direction);								
	}
	
	/**
	 * Invert rayThrough operation, that is return a vector whose first two indecies are the row and column and the last is the distance from the eye in the n direction
	 * this should be very close to 1 and can be used as a measure of accuracy.
	 * 
	 * @param the ray that goes through the view-plane and the eye
	 * 
	 * @return the vector described above
	 */
	public Vector invertRayThrough(Ray r) {
		r.setDirection(r.getDirection().normalize());//start of with a normalized ray
		return invertNormalizedRayThrough(r);
		
	}
	private Vector invertNormalizedRayThrough(Ray r) {
		
		Vector nPart = Vector.fromColumnMatrix(n.scalarMultiply(-N));
		Vector uPart = Vector.fromColumnMatrix(u.scalarMultiply(W));
		Vector vPart = Vector.fromColumnMatrix(v.scalarMultiply(H));

		Matrix m = uPart.createAugmentedMatrix(vPart).createAugmentedMatrix(nPart).createAugmentedMatrix(r.getDirection());
		m.setFirstColumnIndex(1);
		m.setFirstRowIndex(1);
		m = Matrix.createRREF(m);

		Vector v = m.getColumn(4);
		v = v.add(new Point(1d,1d,0d));//since we are dealing with positive and negative percents we push all our important values over so that the resulting number is 2x the desired percent of the width
		v = new Point(
				this.getPlaneWidth()*v.getValueAt(1)/2d//convert to real percents and multiply by the desired length
				,this.getPlaneHeight()*v.getValueAt(2)/2d,
				v.getValueAt(3));
		if(Math.abs(v.getValueAt(3))-1d>.0000001) {//the direction was not on the projection plane
			r.setDirection(Vector.fromColumnMatrix(r.getDirection().scalarMultiply(1d/v.getValueAt(3))));
			return invertNormalizedRayThrough(r);
		}
		return v;		
	}
	
	public Point getLocation () {
		return new Point (eye.getX(), eye.getY(), eye.getZ());
	}
	public double getAngle() {		
		return 180d/Math.PI*Math.acos(eye.dot(lookAt)/(eye.length()*lookAt.length()));				
	}
	
	/**
	 * Returns the width of the viewplane
	 * @return an integer representing the number of columns in the view plane
	 */
	public int getPlaneWidth () {
		return planeSize.width;
	}
	
	/**
	 * Returns the height of the viewplane
	 * @return an integer representing the number of rows in the view plane
	 */
	public int getPlaneHeight () {
		return planeSize.height;		
	}
	
	/**
	 * Rotates the camera around its z axis by the specified amount
	 * @param angle the angle, in degrees, to roll the camera by
	 */
	public void roll (double angle) {
		
	}
	
	/**
	 * Rotates the camera around its x axis by the specified amount
	 * @param angle the angle, in degrees, to pitch the camera by
	 */
	public void pitch (double angle) {
		
	}
	
	/**
	 * Rotates the camera aroud its y axis by the specified amount
	 * @param angle the angle, in degrees to pitch the camera by
	 */
	public void yaw (double angle) {
		
	}
	
	/**
	 * Moves the camera along its z axis
	 * @param amount the distance to move the camera
	 */
	public void slide (double amount) {
		
	}

	/**  This method is used by the Log4J system when a logger attempts to render an object  **/
	public String doRender(Object arg0) {
		StringBuffer buffer = new StringBuffer (256);
		buffer.append ("Camera Location: ");
		buffer.append (eye);
		buffer.append ("\n");
		
		buffer.append ("Looking at: ");
		buffer.append (lookAt);
		buffer.append ("\n");
		
		buffer.append ("Distance");
		buffer.append (this.N);
		
		buffer.append ("Plane size (width x height):");
		buffer.append (getPlaneWidth());
		buffer.append (" x ");
		buffer.append (getPlaneHeight());
		buffer.append ("\n");
		
		buffer.append ("u axis:");
		buffer.append (u.toString());
		buffer.append ("\nv axis:");
		buffer.append (v.toString());
		buffer.append ("\nn axis:");
		buffer.append (n.toString());
		
		
		return buffer.toString();
		
	}
	
	public String toString() {
		return doRender(this);
	}

	public Vector getN() {
		return n;
	}

	public Vector getU() {
		return u;
	}

	public Vector getV() {
		return v;
	}
	public double getViewplaneDistance() {
		return this.N;
	}

	public void setPlaneSize(Dimension planeSize) {
		this.planeSize = planeSize;
	}
}
